package gov.va.med.mhv.usermgmt.service.impl;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.BooleanServiceResponse;
import org.tigris.atlas.service.StringServiceResponse;
import org.tigris.atlas.service.VoidServiceResponse;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.TimestampUtils;
import gov.va.med.mhv.service.MHVAbstractService;
import gov.va.med.mhv.usermgmt.bizobj.ExtractControlBO;
import gov.va.med.mhv.usermgmt.enumeration.ExtractEnablementStatus;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.integration.adapter.PatientExtractServiceAdapter;
import gov.va.med.mhv.usermgmt.service.ExtractService;
import gov.va.med.mhv.usermgmt.service.FacilityExtractStatusCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientExtractStatusServiceResponse;
import gov.va.med.mhv.usermgmt.transfer.Facility;
import gov.va.med.mhv.usermgmt.transfer.FacilityExtractStatus;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.PatientExtractStatus;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.util.ExtractEnablementStatusUtils;
import gov.va.med.mhv.usermgmt.util.PHRAccessControl;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;

/**
 * Service implementation class for the Extract service
 * @see gov.va.med.mhv.usermgmt.service.ExtractService
 */
public class ExtractServiceImpl extends MHVAbstractService
    implements ExtractService
{

    private static final Log LOG = LogFactory.getLog(ExtractServiceImpl.class);

    private PatientExtractServiceAdapter adapter =
        new PatientExtractServiceAdapter();
    /**
     * Should return one of the values defined by the ExtractEnablementStatus
     * enumeration. Note that this is returned as a String
     * @return the string value of the ExtractEnablementStatus
     *
     */
	  private static String getExtractEnablement(ExtractType extractType) {
		  Precondition.assertNotNull("extractType", extractType);
	  	  List controls = ExtractControlBO.getExtractEnabledByType(
	  			extractType.getValue());
	  	  if (controls != null) {
	  		for(Object c: controls) {
	  			ExtractControlBO control = (ExtractControlBO) c;
	  			if ((control != null)
	  					&& (control.getEnablementStatus() ==
	  					ExtractEnablementStatusUtils.ENABLED))
	  			{
	  				return ExtractEnablementStatus.ENABLED;
	  			} else if((control != null)
						&& (control.getEnablementStatus() ==
							ExtractEnablementStatusUtils.PARTIAL)){
	  					return ExtractEnablementStatus.PARTIAL;
	  				}
	  			}
	  		}
	  		//If we cannot get the actual value then return disabled
	  		 if (LOG.isWarnEnabled()) {
	                LOG.warn("Skipping retrieval for " + extractType + ". " +
	                		"Setting default to Disabled"
	                		 + "Note: that the operation attempted a lookup of" +
	                		   " an extract type that does not" +
	                		   "exist!");
	            }

	  	return ExtractEnablementStatus.DISABLED;
	}


    private static boolean isInProgress(
        Collection<FacilityExtractStatus> statusses)
    {
        for (FacilityExtractStatus status: statusses) {
            if (BooleanUtils.isTrue(status.getInProgress())) {
                return true;
            }
        }
        return false;
    }

    private static Date calculateLastUpdated(
        Collection<FacilityExtractStatus> statusses)
    {
        Timestamp lastUpdated = null;
        for (FacilityExtractStatus status: statusses) {
            if ((lastUpdated == null)
                || TimestampUtils.before(lastUpdated, status.getLastUpdated()))
            {
                lastUpdated = status.getLastUpdated();
            }
        }
        return (lastUpdated != null) ? new Date(lastUpdated.getTime()) : null;
    }

    /* (non-Javadoc)
	 * @see gov.va.med.mhv.usermgmt.service.impl.ExtractServiceTest#update(
     * 			gov.va.med.mhv.usermgmt.enumeration.ExtractType,
     *			gov.va.med.mhv.usermgmt.transfer.Patient)
	 */
    public VoidServiceResponse update(ExtractType extractType,
        Patient patient)
    {
        Precondition.assertNotNull("extractType", extractType);
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.userProfile", patient.
            getUserProfile());


		if (!PatientCorrelationStatusUtils.isCorrelated(patient)) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Skipping update for " + extractType + " and "
                    + patient.getUserProfile().getUserName()
                    + ", because patient is not correlated.");
            }
            return new VoidServiceResponse();
        } else if (!hasAccess(extractType, patient)) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Skipping update for " + extractType + " and "
                    + patient.getUserProfile().getUserName()
                    + ", because patient has no access to extract.");
            }
            return new VoidServiceResponse();
        } else if (!(getExtractEnablement(extractType).equalsIgnoreCase(
            ExtractEnablementStatus.ENABLED)))
        {
            if (LOG.isInfoEnabled()) {
                LOG.info("Skipping update for " + extractType + " and " + 
                    patient.getUserProfile().getUserName() 
                    + ", because extract is disabled or partial-read-only.");
            }
            return new VoidServiceResponse();
        } 
	        //By isExtractEnabled the application is allowing the user
	        //to retrieve their new data from Vista.
        return adapter.updateExtract(patient, getStationNumbers(patient));
    }

    private List<String> getStationNumbers(Patient patient) {
        List<String> stationNumbers = new ArrayList<String>();
        for (Object f: patient.getFacilitys()) {
            // Note that the name attribute of
            // Facility represents the station number
            stationNumbers.add(((Facility) f).getName());
        }
        return stationNumbers;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.ExtractService#getPatientStatus(
     * gov.va.med.mhv.usermgmt.enumeration.ExtractType,
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    @SuppressWarnings("unchecked")
    public StringServiceResponse getExtractViewStatus(ExtractType extractType) {
        Precondition.assertNotNull("extractType", extractType);
        StringServiceResponse response = new StringServiceResponse();
        response.setString(getExtractEnablement(extractType));
        return response;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.ExtractService#getPatientStatus(
     * gov.va.med.mhv.usermgmt.enumeration.ExtractType,
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    @SuppressWarnings("unchecked")
    public PatientExtractStatusServiceResponse
        getPatientStatus(ExtractType extractType, Patient patient)
    {
        Precondition.assertNotNull("extractType", extractType);
        Precondition.assertNotNull("patient", patient);

        PatientExtractStatusServiceResponse response =
            new PatientExtractStatusServiceResponse();
        response.setPatientExtractStatus(null);
        FacilityExtractStatusCollectionServiceResponse statusResponse =
            getFacilityStatus(extractType, patient);
        if (hasErrorMessages(statusResponse)) {
            copyMessages(response, statusResponse);
        } else {
            PatientExtractStatus status = TransferObjectFactory.
                createPatientExtractStatus();
            status.setInProgress(isInProgress(statusResponse.getItems()));
            status.setLastUpdated(calculateLastUpdated(statusResponse.
                getItems()));
            status.setPatientId(patient.getId());
            status.setExtractType(extractType);
            response.setPatientExtractStatus(status);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("getPatientStatus for " + describe(patient) + " and " +
                describe(extractType) + " yields " +
                describe(response.getPatientExtractStatus()));
        }

        return response;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.ExtractService#getFacilityStatus(
     * gov.va.med.mhv.usermgmt.enumeration.ExtractType,
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    public FacilityExtractStatusCollectionServiceResponse
        getFacilityStatus(ExtractType extractType, Patient patient)
    {
        Precondition.assertNotNull("extractType", extractType);
        Precondition.assertNotNull("patient", patient);

        FacilityExtractStatusCollectionServiceResponse response =
            new FacilityExtractStatusCollectionServiceResponse();

        response = adapter.getExtractStatus(extractType, patient);

        return response;
    }
    
    @SuppressWarnings("unchecked")
    private boolean hasAccess(ExtractType extractType, Patient patient) {
        BooleanServiceResponse response = PHRAccessControl.hasPhrAccess(
            extractType, patient);
        return !hasErrorMessages(response) && BooleanUtils.isTrue(response.
            getBoolean()); 
    }

    private static final String describe(Patient patient) {
        return ((patient != null) && (patient.getUserProfile() != null))
            ? "patient '" + patient.getUserProfile().getUserName() + "'"
            : "<unknown patient>";
    }
    private static final String describe(ExtractType extractType) {
        return (extractType != null)
            ? "'" + extractType.getName() + "' extract"
            : "<unknown extract>";
    }
    private static String describe(PatientExtractStatus status) {
        StringBuilder builder = new StringBuilder();
        if (status != null) {
            builder.append("PatientExtractStatus@");
            builder.append(Integer.toHexString(status.hashCode()));
            builder.append("[ExtractType=");
            builder.append(describe(status.getExtractType()));
            builder.append(", PatientId=");
            builder.append(status.getPatientId());
            builder.append(", InProgress=");
            builder.append(status.getInProgress());
            builder.append(", LastUpdated=");
            builder.append(status.getLastUpdated());
            builder.append("]");
        } else {
            builder.append("<No PatientExtractStatus>");
        }
        return builder.toString();
    }
}